{========================================================================}
{=                (c) 1995-98 SwiftSoft Ronald Dittrich                 =}
{========================================================================}
{=                          All Rights Reserved                         =}
{========================================================================}
{=  D 01099 Dresden             = Fax.: +49 (0)351-8037944              =}
{=  Loewenstr.7a                = info@swiftsoft.de                     =}
{========================================================================}
{=  Actual versions on http://www.swiftsoft.de/mmtools.html             =}
{========================================================================}
{=  This code is for reference purposes only and may not be copied or   =}
{=  distributed in any format electronic or otherwise except one copy   =}
{=  for backup purposes.                                                =}
{=                                                                      =}
{=  No Delphi Component Kit or Component individually or in a collection=}
{=  subclassed or otherwise from the code in this unit, or associated   =}
{=  .pas, .dfm, .dcu, .asm or .obj files may be sold or distributed     =}
{=  without express permission from SwiftSoft.                          =}
{=                                                                      =}
{=  For more licence informations please refer to the associated        =}
{=  HelpFile.                                                           =}
{========================================================================}
{=  $Date: 06.09.98 - 14:17:35 $                                        =}
{========================================================================}
unit Unit3;

interface
         
uses
  Windows,
  SysUtils,
  Messages,
  Classes,
  Graphics,
  Controls,
  Forms,
  Dialogs,
  StdCtrls,
  ExtCtrls,
  FileCtrl,
  Menus,
  ComCtrls,
  MMMPType,
  MMMpeg,
  MMWaveIO,
  MMRiff;

type
  TPlaylistEditor = class(TForm)
    OKButton: TButton;
    CancelButton: TButton;
    MainMenu1: TMainMenu;
    File1: TMenuItem;
    Label5: TLabel;
    Label6: TLabel;
    PlayListBox: TListBox;
    AddButton: TButton;
    RemoveButton: TButton;
    ClearButton: TButton;
    RandomizeButton: TButton;
    FileListBox: TFileListBox;
    DirectoryListBox1: TDirectoryListBox;
    DriveComboBox1: TDriveComboBox;
    Bevel1: TBevel;
    NewPlaylist1: TMenuItem;
    OpenPlaylist1: TMenuItem;
    SavePlaylist1: TMenuItem;
    SavePlaylistAs1: TMenuItem;
    AppendPlaylist1: TMenuItem;
    N1: TMenuItem;
    Exit1: TMenuItem;
    OpenDialog: TOpenDialog;
    SaveDialog: TSaveDialog;
    InfoLabel: TLabel;
    procedure ListMouseUp(Sender: TObject; Button: TMouseButton;
                          Shift: TShiftState; X, Y: Integer);
    procedure ListKeyUp(Sender: TObject; var Key: Word;
                        Shift: TShiftState);
    procedure FormShow(Sender: TObject);
    procedure ListDrawItem(Control: TWinControl; Index: Integer;
                           Rect: TRect; State: TOwnerDrawState);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ClearButtonClick(Sender: TObject);
    procedure RandomizeButtonClick(Sender: TObject);
    procedure AddButtonClick(Sender: TObject);
    procedure RemoveButtonClick(Sender: TObject);
    procedure ListDragOver(Sender, Source: TObject; X, Y: Integer;
                           State: TDragState; var Accept: Boolean);
    procedure ListDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure ListEndDrag(Sender, Target: TObject; X, Y: Integer);
    procedure FormHide(Sender: TObject);
    procedure Exit1Click(Sender: TObject);
    procedure NewPlaylist1Click(Sender: TObject);
    procedure OpenPlaylist1Click(Sender: TObject);
    procedure SavePlaylist1Click(Sender: TObject);
    procedure SavePlaylistAs1Click(Sender: TObject);
    procedure AppendPlaylist1Click(Sender: TObject);
  private
    FListName : TFileName;
    DragTarget: TListBox;
    aTimer    : TTimer;
    aBitmap1  : TBitmap;
    aBitmap2  : TBitmap;
    aIcon     : TIcon;
    oldIndex  : integer;
    oldCaption: String;

    procedure CreateParams(var Params: TCreateParams); override;
    procedure SetListName(aValue: TFileName);
    
    procedure SetButtons;
    function  FirstSelection(aList: TCustomListBox): Integer;
    function  LastSelection(aList: TCustomListBox): Integer;
    function  FindIndex(aList: TListBox; aPos: TPoint): integer;
    procedure ClearSelected(aList: TCustomListBox);
    procedure AddSelected(aIndex: integer);
    procedure ResortSelected(aIndex: integer);
    procedure RemoveSelected;
    procedure DrawIndexPtr(oldIndex, newIndex: integer);
    procedure DragTimerExpired(Sender: TObject);
    procedure UpdatePlayListBox;
    procedure SetFileInfo;
  public
    TempPlayList: TStringList;
    ListChanged: Boolean;

    property ListName: TFileName read FListName write SetListName;
  end;

var
  PlaylistEditor: TPlaylistEditor;

function LoadPlayList(FileName: TFileName; aPlayList: TStringList): Boolean;
function SavePlayList(FileName: TFileName; aPlayList: TStringList): Boolean;

implementation

uses unit1, form_mainform;

{$R *.DFM}

const
  crTrackDrag   = 1;
  crTrackAdd    = 2;
  crTrackDelete = 3;

{------------------------------------------------------------------------}
function LoadPlayList(FileName: TFileName; aPlayList: TStringList): Boolean;
var
   i: integer;
   F: TextFile;
   S: String;

begin
   i := 0;
   if (FileName <> '') and FileExists(FileName) then
   begin
      AssignFile(F, FileName);
      {$I+}
      Reset(F);
      try
         while not EOF(F) do
         begin
            ReadLn(F, S);
            if (S <> '') then
            begin
               if FileExists(S) then
               begin
                  if IsMpegFile(S) or wioIsWaveFile(S, RIFF_FILE) then
                  begin
                     aPlayList.Add(S);
                     inc(i);
                  end
                  else
                     MessageDlg(S+' is not a valid Audiofile',mtError, [mbOK],0);
               end;
            end;
         end;

      finally
         CloseFile(F);
      end;
      {$I+}
   end;

   Result := (i > 0);
end;

{------------------------------------------------------------------------}
function SavePlayList(FileName: TFileName; aPlayList: TStringList): Boolean;
var
   i: integer;
   F: TextFile;

begin
   Result := True;
   if (FileName <> '') then
   begin
      AssignFile(F, FileName);
      {$I-}
      Rewrite(F);
      try
         if (IOResult <> 0) then Result := False
         else
         begin
            for i := 0 to aPlayList.Count-1 do
            begin
               WriteLn(F,aPlayList[i]);
               if (IOResult <> 0) then
               begin
                  Result := False;
                  break;
               end;
            end;
         end;
      finally
         CloseFile(F);
      end;
      {$I+}
   end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlayListEditor.CreateParams(var Params: TCreateParams);
begin
   inherited CreateParams(Params);
   Params.Style := Params.Style and not WS_SIZEBOX;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.FormCreate(Sender: TObject);
begin
     oldCaption := Caption;

     TempPlayList := TStringList.Create;
     ListName := 'noname.m3u';

     aTimer := TTimer.Create(Self);
     aTimer.Interval := 50;
     aTimer.Enabled := False;
     aTimer.OnTimer := DragTimerExpired;

     aBitmap1 := TBitmap.Create;
     aBitmap2 := TBitmap.Create;
     aBitmap1.Handle := LoadBitmap(HInstance, 'BM_NOTE');
     aBitmap2.Width := aBitmap1.Width;
     aBitmap2.Height := aBitmap1.Height;
     BitBlt(aBitmap2.Canvas.Handle, 0,0, aBitmap1.Width, aBitmap1.Height,
            aBitmap1.Canvas.Handle, 0,0, NOTSRCCOPY);

     aIcon := TIcon.Create;
     aIcon.Handle := LoadIcon(HInstance, 'MARKERICON');

     Icon.Handle := LoadIcon(HInstance, 'PLAYLISTICON');

     oldIndex := -1;

     Screen.Cursors[crTrackDrag] := LoadCursor(HInstance, 'CR_TRACKDRAG');
     Screen.Cursors[crTrackAdd] := LoadCursor(HInstance, 'CR_TRACKADD');
     Screen.Cursors[crTrackDelete] := LoadCursor(HInstance, 'CR_TRACKDELETE');

     DragTarget := Nil;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.FormDestroy(Sender: TObject);
begin
     aTimer.Free;
     aBitmap1.Free;
     aBitmap2.Free;
     aIcon.Free;
     TempPlayList.Free;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.FormShow(Sender: TObject);
begin
     PlayListBox.Clear;
     TempPlayList.Assign(mainplayer.PlayList);
     ListName := mainplayer.PlayListName;
     UpdatePlayListBox;
     ListChanged := not ((ListName <> '') and (ListName <> 'noname.m3u'));

     SetFileInfo;
     SetButtons;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.FormHide(Sender: TObject);
begin
   if ModalResult = mrOK then
   begin
      if ListChanged then
         if MessageDlg('Save Changes to the PlayList ?', mtConfirmation, [mbYes,mbNo],0) = mrYes then
         begin
            if (ListName <> 'noname.m3u') then
                SavePlaylist1Click(nil)
            else
                SavePlaylistAs1Click(nil);
         end;
   end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlayListEditor.SetListName(aValue: TFileName);
begin
   FListName := aValue;
   Caption := oldCaption;
   if FListName <> '' then
      Caption := Caption + ' - ' + FListName;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.SetButtons;
begin
     AddButton.Enabled       := (FileListBox.SelCount > 0);
     RemoveButton.Enabled    := (PlayListBox.SelCount > 0);
     ClearButton.Enabled     := (PlayListBox.Items.Count > 0);
     RandomizeButton.Enabled := (PlayListBox.Items.Count > 0);
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlayListEditor.UpdatePlayListBox;
var
   i: integer;
begin
   PlayListBox.Clear;
   for i := 0 to TempPlayList.Count-1 do
       PlayListBox.Items.Add(ExtractFileName(TempPlayList[i]));
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlayListEditor.SetFileInfo;
var
   FileName,S: String;
   lpwio: PWaveIOCB;
   MpegInfo: TMpegFileInfo;

begin
   if (TempPlayList.Count > 0) and (PlayListBox.ItemIndex >= 0) then
   begin
      FileName := TempPlayList[PlayListBox.ItemIndex];
      if IsMpegFile(FileName) then
      begin
         if GetMpegFileInfo(FileName, MpegInfo) then
         begin
            S := 'ISO MPEG ';
            with MpegInfo do
            begin
               case Version of
                 v1   : S := S + '1 - ';
                 v2LSF: S := S + '2 - ';
               end;
               case Layer of
                    1: S := S + 'Layer I; ';
                    2: S := S + 'Layer II; ';
                    3: S := S + 'Layer III; ';
               end;
               S := S + Format('%2.3f kHz; %d KBit/s; ',[SampleRate/1000,BitRate]);
               case SampleMode of
                 smStereo       : S := S + 'Stereo;';
                 smJointStereo  : S := S + 'Joint Stereo;';
                 smDualChannel  : S := S + 'Dual Channel;';
                 smSingleChannel: S := S + 'Mono;';
               end;
               InfoLabel.Caption := S;
            end;
            exit;
         end;
      end
      else if wioIsWaveFile(FileName, RIFF_FILE) then
      begin
         if wioReadFileInfo(lpwio, PChar(FileName),
                            mmioFOURCC('W', 'A', 'V', 'E'), RIFF_FILE) = 0 then
         try
            InfoLabel.Caption := 'WAVE - ';
            wioGetFormatName(@lpwio.wfx, S);
            InfoLabel.Caption := InfoLabel.Caption+' '+S;
            wioGetFormat(@lpwio.wfx, S);
            InfoLabel.Caption := InfoLabel.Caption+' '+S;
            exit;
         finally
            wioFreeFileInfo(lpwio);
         end;
      end;
      InfoLabel.Caption := 'Unknown File';
   end
   else InfoLabel.Caption := 'Empty';
end;

{-- TPlayListEditor -----------------------------------------------------}
function TPlaylistEditor.FirstSelection(aList: TCustomListBox): Integer;
begin
     for Result := 0 to aList.Items.Count - 1 do
         if aList.Selected[Result] then exit;
     Result := LB_ERR;
end;

{-- TPlayListEditor -----------------------------------------------------}
function TPlaylistEditor.LastSelection(aList: TCustomListBox): Integer;
begin
     for Result := aList.Items.Count - 1 downTo 0 do
         if aList.Selected[Result] then exit;
     Result := LB_ERR;
end;

{-- TPlayListEditor -----------------------------------------------------}
function  TPlaylistEditor.FindIndex(aList: TListBox; aPos: TPoint): integer;
begin
     with aList do
     begin
          Result := ItemAtPos(aPos, False);
          if Items.Count > (Height div ItemHeight)-1 then
             if Result = TopIndex + (Height div ItemHeight)-1 then
                   if aPos.Y > Height-(ItemHeight div 2) then
                      inc(Result);
     end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.ClearSelected(aList: TCustomListBox);
Var
   aIndex: integer;

begin
     aIndex := FirstSelection(aList);
     if aIndex > LB_Err then
     begin
        while aIndex <= LastSelection(aList) do
        begin
             if aList.Selected[aIndex] then
             begin
                aList.Selected[aIndex] := False;
                ListChanged := True;
             end;
             inc(aIndex);
        end;
        SetFileInfo;
     end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.AddSelected(aIndex: integer);
Var
  i: Integer;

begin
     with TempPlayList do
     begin
          if (aIndex = -1) then aIndex := Count;

          for i := 0 to FileListBox.Items.Count - 1 do
          begin
             if FileListBox.Selected[i] then
             begin
                Insert(aIndex, FileListBox.Items[i]);
                ListChanged := True;
                inc(aIndex);
             end;
          end;

          UpdatePlayListBox;

          if aIndex >= PlayListBox.Height div PlayListBox.ItemHeight then
             PlayListBox.TopIndex := aIndex-((PlayListBox.Height div PlayListBox.ItemHeight)-1);
     end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.ResortSelected(aIndex: integer);
Var
  i: Integer;

begin
     if (PlayListBox.Items.Count > 1) then
     with PlayListBox do
     begin
          if (aIndex = -1) then aIndex := 0;

          i := 0;
          while i < Items.Count do
          begin
              if Selected[i] then
              begin
                   Selected[i] := False;
                   ListChanged := True;
                   if aIndex > i then
                   begin
                        TempPlayList.Move(i, aIndex-1);
                        Items.Move(i, aIndex-1);
                        dec(i);
                   end
                   else
                   begin
                        TempPlayList.Move(i, aIndex);
                        Items.Move(i, aIndex);
                        inc(aIndex);
                   end;
              end;
              inc(i);
          end;

          if (Items.Count > 0) then
          begin
               TopIndex := 0;
               Selected[0] := True;
               Selected[0] := False;
          end;
     end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.RemoveSelected;
Var
  i: Integer;

begin
     with PlayListBox do
     begin
          ItemIndex := 0;
          for i := Items.Count - 1 downTo 0 do
              if Selected[i] then
              begin
                 Items.Delete(i);
                 TempPlayList.Delete(i);
                 ListChanged := True;
              end;

          if (Items.Count > 0) then
          begin
               TopIndex := 0;
               Selected[0] := True;
               Selected[0] := False;
          end;
     end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.ListMouseUp(Sender: TObject;
          Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
     if Button = mbLeft then
     begin
          if (Sender = PlayListBox) then SetFileInfo;
          SetButtons;
     end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.ListKeyUp(Sender: TObject; var Key: Word;
                                          Shift: TShiftState);
begin
     if (Sender = PlayListBox) then SetFileInfo;
     SetButtons;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.ListDrawItem(Control: TWinControl;
          Index: Integer; Rect: TRect; State: TOwnerDrawState);
Var
  Offset: Integer;

begin
     with (Control as TListBox), (Control as TListBox).Canvas do
     begin
          FillRect(Rect);
          Offset := 1;

          if (odSelected in State) then
             BrushCopy(Bounds(Rect.Left + Offset, Rect.Top,
                              aBitmap2.Width, aBitmap2.Height),
                       aBitmap2,
                       Bounds(0, 0, aBitmap2.Width, aBitmap2.Height),
                       clBlack)
          else
             BrushCopy(Bounds(Rect.Left + Offset, Rect.Top,
                              aBitmap1.Width, aBitmap1.Height),
                       aBitmap1,
                       Bounds(0, 0, aBitmap1.Width, aBitmap1.Height),
                       clWhite);

          Offset := Offset + aBitmap1.Width + 5;
          TextOut(Rect.Left + Offset, Rect.Top, Items[Index]);
     end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.AddButtonClick(Sender: TObject);
begin
     AddSelected(-1);
     SetButtons;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.RemoveButtonClick(Sender: TObject);
begin
     RemoveSelected;
     SetButtons;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.ClearButtonClick(Sender: TObject);
begin
     TempPlayList.Clear;
     PlayListBox.Clear;
     ListChanged := True;
     SetFileInfo;
     SetButtons;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.RandomizeButtonClick(Sender: TObject);
var
   i,j: integer;
begin
     Randomize;
     for i := 0 to TempPlayList.Count-1 do
     begin
        j := Random(TempPlayList.Count);
        TempPlayList.Move(i, j);
        PlayListBox.Items.Move(i, j);
     end;
     ListChanged := True;
     SetButtons;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.DragTimerExpired(Sender: TObject);

Var
   MousePos: TPoint;

begin
     if DragTarget <> Nil then
     begin
        GetCursorPos(MousePos);
        MousePos := ScreenToClient(MousePos);

        with DragTarget do
        begin
           if (MousePos.X > Left) And (MousePos.X < Left + Width) then
           begin
              { scroll the listbox up }
              if (MousePos.Y < Top) And (TopIndex > 0) then
                 TopIndex := TopIndex - 1
              else
              { scroll the listbox down }
              if (MousePos.Y > Top + Height) And (TopIndex < Items.Count - (Height div ItemHeight)) then
                 TopIndex := TopIndex + 1;
           end;
        end;
     end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.DrawIndexPtr(oldIndex, newIndex: integer);
const
     Offset: integer = 2;

begin
     with Canvas do
     begin
          if oldIndex <> LB_Err then
          begin
               with PlayListBox do
                  oldIndex := (oldIndex - TopIndex) * ItemHeight + Top - 5;
               Brush.Color := Self.Color;
               FillRect(Rect(Offset,oldIndex,
                             Offset+15,
                             oldIndex+15));
          end;
          if newIndex <> LB_Err then
          begin
               with PlayListBox do
                  newIndex := (newIndex - TopIndex) * ItemHeight + Top - 5;
               Draw(Offset, newIndex, aIcon);
          end;
     end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.ListDragOver(Sender, Source: TObject;
                        X, Y: Integer; State: TDragState;
                        Var Accept: Boolean);
Var
   curIndex: integer;

begin
     if (Source is TCustomListBox) And (Sender is TCustomListBox) then
     begin
          Accept := True;

          { set the right drag cursors }
          if (State = dsDragEnter) then
          begin
             if Source = PlayListBox then
             begin
                if Sender = PlayListBox then
                   TListBox(Source).DragCursor := crTrackDrag
                else
                   TFileListBox(Source).DragCursor := crTrackDelete;
             end
             else
             begin
                if Sender = FileListBox then
                   TFileListBox(Source).DragCursor := crTrackDrag
                else
                   TListBox(Source).DragCursor := crTrackAdd;
             end;

             aTimer.Enabled := False;
             DragTarget := TListBox(Sender);
          end
          else if (State = dsDragLeave) then
                  aTimer.Enabled := True;

          { don't accept if on the scrollbars }
          with TCustomListBox(Sender) do
          begin
             CurIndex := ItemAtPos(Point(X,Y),False);
             if CurIndex = LB_Err then Accept := False;
          end;

          { now draw the index arrow }
          if (Sender = PlayListBox) then
          begin
             {special case for the last visible item }
             CurIndex := FindIndex(TListBox(Sender), Point(X, Y));

             if (CurIndex <> oldIndex) Or (State = dsDragLeave) then
             begin
                if (State = dsDragEnter) then
                   oldIndex := LB_Err;

                if (State = dsDragLeave) then
                   curIndex := LB_Err;

                DrawIndexPtr(oldIndex, curIndex);

                oldIndex := curIndex;
             end;
          end;
     end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.ListDragDrop(Sender, Source: TObject;
                                     X, Y: Integer);
Var
  aIndex: Integer;

begin
    { make sure source and destination components are list boxes }
    if (Source is TCustomListBox) and (Sender is TCustomListBox) then
    begin
         if (Sender = FileListBox) then
         begin
              { delete selected items }
              if (Source = PlayListBox) then
                 RemoveSelected;
         end
         else
         begin
              { copy from one list to another }
              if (Source = FileListBox) then
              begin
                   { find destination position in list box }
                   aIndex := FindIndex(TListBox(Sender), Point(X, Y));
                   AddSelected(aIndex);
              end
              else { rearrange list }
              begin
                   { find destination position in list box }
                   aIndex := FindIndex(TListBox(Sender), Point(X, Y));
                   ResortSelected(aIndex);
              end;
         end;
    end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.ListEndDrag(Sender, Target: TObject;
                                    X, Y: Integer);
begin
     aTimer.Enabled := False;
     DragTarget := Nil;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.Exit1Click(Sender: TObject);
begin
   Close;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.NewPlaylist1Click(Sender: TObject);
begin
   if SaveDialog.Execute then
   begin
      TempPlayList.Clear;
      PlayListBox.Clear;
      ListChanged := True;

      ListName := SaveDialog.FileName;
   end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.OpenPlaylist1Click(Sender: TObject);
begin
   if OpenDialog.Execute then
   begin
      TempPlayList.Clear;
      PlayListBox.Clear;

      if LoadPlayList(OpenDialog.FileName,TempPlayList) then
      begin
         UpdatePlayListBox;
         ListChanged := False;
         ListName := OpenDialog.FileName;
      end
      else MessageDlg('Unable to load Playlist!',mtError, [mbOK],0);
   end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.SavePlaylist1Click(Sender: TObject);
begin
   if SavePlayList(ListName,TempPlayList) then
   begin
      ListChanged := False;
   end
   else MessageDlg('Unable to save Playlist!',mtError, [mbOK],0);
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.SavePlaylistAs1Click(Sender: TObject);
begin
   SaveDialog.InitialDir := ExtractFilePath(ListName);
   SaveDialog.FileName := ExtractFileName(ListName);

   if SaveDialog.Execute then
   begin
      if SavePlayList(SaveDialog.FileName,TempPlayList) then
      begin
         ListChanged := False;
         ListName := SaveDialog.FileName;
      end
      else MessageDlg('Unable to save Playlist!',mtError, [mbOK],0);
   end;
end;

{-- TPlayListEditor -----------------------------------------------------}
procedure TPlaylistEditor.AppendPlaylist1Click(Sender: TObject);
begin
   if OpenDialog.Execute then
   begin
      if LoadPlayList(OpenDialog.FileName,TempPlayList) then
      begin
         UpdatePlayListBox;
         ListChanged := True;
      end
      else MessageDlg('Unable to append Playlist!',mtError, [mbOK],0);
   end;
end;

end.
